using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Media;
using System.Text;
using System.Threading;
using System.Windows.Forms;
using System.Text.RegularExpressions;
using Psion.RFID;
using Psion.RFID.LF;

namespace LF_DEMO
{
    public partial class Main : Form
    {
        private readonly Reader _reader;
        private readonly RFIDDriver _rfidDriver;
        private SoundPlayer _beepinventory = new SoundPlayer(new MemoryStream(Properties.Resources.Beep_Inventory));

        public Main()
        {
            InitializeComponent();

            _textBoxRWNrBytes.Text = "2";
            _textBoxRWOffset.Text = "0";

            try
            {
                // instantiation of RFID driver
                _rfidDriver = new RFIDDriver();

                // check if the driver is installed
                if (!_rfidDriver.IsInstalled)
                    throw new Exception("RFID Driver is not installed");

                // check and enable the driver
                //  it means to power ON the module 
                //  assign com port number
                if (! _rfidDriver.IsEnabled) _rfidDriver.Enable();

                // necessary time to load the driver
                Thread.Sleep(250);
            }
            catch (Exception exception)
            {
                MessageBox.Show("Error by enabling the RFID driver\n" + exception.Message);
                Application.Exit();
            }

            // NOTE: RFID Driver has to be enabled before to try to connect to the LF reader

            try
            {
                // instantiation of LF reader
                _reader = new Reader();

                // open the communication with the reader
                EERROR eerror = _reader.OpenReader(_rfidDriver.ComPort);
                if (eerror != EERROR.ER_OK) throw new Exception("Open reader failed: " + eerror.ToString());

                // check the dialog with the reader by getting the FW version
                byte[] arrReaderVer = null;
                eerror = _reader.GetReaderVersion(ref arrReaderVer);
                if (eerror != EERROR.ER_OK) throw new Exception("Dialog with reader failed: " + eerror.ToString());

            }
            catch (Exception exception)
            {
                MessageBox.Show("Error by opening the connection with the LF module\n" + exception.Message);
                Application.Exit();
            }
        }

        private void Main_Closing(object sender, System.ComponentModel.CancelEventArgs e)
        {
            // stop continuous read if it was not stopped
            if (_timerReadCont.Enabled) _timerReadCont.Enabled = false;

            // close reader connection
            if (_reader != null) _reader.CloseReader();

            // disable rfid driver
            //  it means to destroy com port created
            //  and turn OFF the power on the module
            if (_rfidDriver != null && _rfidDriver.IsInstalled && _rfidDriver.IsEnabled)
                _rfidDriver.Disable();
        }


        private void _buttonInfoInfo_Click(object sender, EventArgs e)
        {
            _buttonInfoInfo.Enabled = false;
            byte[] arrData = null;
            // get FW version
            if (_reader.GetReaderVersion(ref arrData) == EERROR.ER_OK)
            {
                // display the fw version
                _textBoxInfoVersion.Text = Encoding.UTF8.GetString(arrData, 0, arrData.Length);
            }
            else MessageBox.Show("Error by getting the FW version");

            arrData = null;
            // get serial number
            if (_reader.GetSerialNumber(ref arrData) == EERROR.ER_OK)
            {
                _textBoxInfoSerianNr.Text = HexToString(arrData);
            }
            else MessageBox.Show("Error by getting the serial number");

            _buttonInfoInfo.Enabled = true;
        }

        private void _buttonInfoReset_Click(object sender, EventArgs e)
        {
            _buttonInfoReset.Enabled = false;
            // reset the reader
            _reader.ResetReader();
            _buttonInfoReset.Enabled = true;
        }

        private void _buttonScanCardID_Click(object sender, EventArgs e)
        {
            _buttonScanCardID.Enabled = false;
            _listScanCard.Items.Clear();
            byte[] arrCardID = null;
            // get tag ID - 's' command
            if (_reader.GetCardID(ref arrCardID) == EERROR.ER_OK)
            {
                // there is no tag in the field
                if (arrCardID.Length == 1 && (char) arrCardID[0x00] == 'N')
                {
                    _listScanCard.Items.Insert(0, new ListViewItem(new[] {"No tag", "in the field"}));
                }
                else
                {
                    // get tag type
                    string strCardType = GetTagType((char) arrCardID[0x00]);

                    // get tag ID
                    byte[] arrCardIDInfo = new byte[arrCardID.Length - 0x01];
                    Array.Copy(arrCardID, 1, arrCardIDInfo, 0, arrCardIDInfo.Length);

                    // display those information
                    string[] arrItem = new string[2]
                                           {
                                               strCardType, Encoding.UTF8.GetString(arrCardIDInfo, 0, arrCardIDInfo.Length)
                                           };
                    _listScanCard.Items.Insert(0, new ListViewItem(arrItem));
                }
            }
            _buttonScanCardID.Enabled = true;
        }

        private void EnableBtn(bool bEnable)
        {
            _buttonScanContCardID.Enabled = bEnable;
            _buttonScanContStop.Enabled = !bEnable;
        }

        private void _buttonScanContCardID_Click(object sender, EventArgs e)
        {
            _listScanContCard.Items.Clear();
            _tags.Clear();
            _tabMain.TabPages.RemoveAt(4);
            _tabMain.TabPages.RemoveAt(3);
            _tabMain.TabPages.RemoveAt(1);
            _tabMain.TabPages.RemoveAt(0);
            // start continuous read
            _reader.SetContinuesReadMode(true);
            _timerReadCont.Enabled = true;

            EnableBtn(false);
        }

        private string GetTagType(char type)
        {
            switch (type)
            {
                case 'h':
                    return "Hitag1/s";
                case 'H':
                    return "Hitag2";
                case 'Q':
                    return "Q5";
                case 'U':
                    return "EM42XX";
                case 'T':
                    return "Titan";
                case 'M':
                    return "Temic";
                case 'Z':
                    return "ISO FDX";
                default:
                    return "Unknown";
            }
        }

        List<Tag> _tags = new List<Tag>();
        public  class Tag
        {
            public string Type;
            public string ID;
            public int Count;
        }

        private void _timerReadCont_Tick(object sender, EventArgs e)
        {
            byte[] arrCardID = null;
            // get continuous read information
            if (_reader.ReadContinuesModeData(ref arrCardID) == EERROR.ER_OK)
            {
                // get tag category (hitag, EM, ...)
                string strCardType = GetTagType((char) arrCardID[0x00]);

                // get tag ID
                byte[] arrCardIDInfo = new byte[arrCardID.Length - 0x01];
                Array.Copy(arrCardID, 1, arrCardIDInfo, 0, arrCardIDInfo.Length);

                string tagID =Encoding.UTF8.GetString(arrCardIDInfo, 0, arrCardIDInfo.Length);

                Tag tag = IsExistingTag(tagID);
                // is new tag 
                if (tag == null)
                {
                    _tags.Add(new Tag() {Type= strCardType, ID = tagID, Count = 1});
                }
                else // increment number of view
                {
                    tag.Count++;
                }
                // display all the tags
                Display();

                // play sound 
                _beepinventory.Play();
            }
        }

        private Tag IsExistingTag(string tagID)
        {
            if (_tags == null || _tags.Count == 0) return null;
            foreach (var tag in _tags) if (tag.ID.Equals(tagID)) return tag;
            return null;
        }
        
        private void Display ()
        {
            _listScanContCard.Items.Clear();
            if (_tags == null || _tags.Count == 0) return;
            foreach (var tag in _tags)
            {
                _listScanContCard.Items.Insert(0, new ListViewItem(
                    new []{tag.Type,tag.ID, tag.Count.ToString()}
                    ));
            }
        }

        private void _buttonScanContStop_Click(object sender, EventArgs e)
        {
            if (_reader != null)
            {
                // stop continuous read
                _timerReadCont.Enabled = false;
                _reader.SetContinuesReadMode(false);
                _tabMain.TabPages.Insert(0, _tabPageInfo);
                _tabMain.TabPages.Insert(1, _tabPageScan);
                _tabMain.TabPages.Insert(3, _tabPageRW);
                _tabMain.TabPages.Insert(4, _tabPageProtocol);

                EnableBtn(true);
            }
        }

        private byte[] StringToByteArray(string str)
        {
            byte[] arrRet = null;

                int iNav = 0;
                arrRet = new byte[str.Length ];
                while (str.Length != iNav)
                {
                    arrRet[iNav ] = Convert.ToByte(str[iNav]);
                    iNav ++;
                }
            
            return arrRet;
        }

        private byte[] HexStringToByteArray(string strHex)
        {
            byte[] arrRet = null;
            if ((strHex.Length%2) == 0)
            {
                int iNav = 0;
                arrRet = new byte[strHex.Length/2];
                while (strHex.Length != iNav)
                {
                    arrRet[iNav/2] = Convert.ToByte(strHex.Substring(iNav, 2), 16);
                    iNav += 0x02;
                }
            }
            return arrRet;
        }

        private string HexToString(byte[] arrData)
        {
            StringBuilder ret = new StringBuilder();
            foreach (byte bData in arrData)
            {
                ret.Append(string.Format("{0:x02}", bData));
            }
            return ret.ToString();
        }

        private void _buttonRWRead_Click(object sender, EventArgs e)
        {
            _buttonRWRead.Enabled = false;
            string strOffset = _textBoxRWOffset.Text;
            string strNrBytes = _textBoxRWNrBytes.Text;
            if ((strOffset.Length > 0) && (strNrBytes.Length > 0))
            {
                byte[] arrCardID = null;
                EERROR eerror = EERROR.ER_OK;

                // get tag ID - 's' command
                if (_reader.GetCardID(ref arrCardID) == EERROR.ER_OK)
                {
                    // there is no tag in the field
                    if (arrCardID.Length == 1 && (char) arrCardID[0x00] == 'N')
                    {
                        MessageBox.Show("No tag in the field");
                    }
                    else
                    {

                        // perform login if the textbox is filled with the password
                        if (_textBoxRWLogin.Text.Length > 0)
                        {
                            byte[] arrLogin = StringToByteArray(_textBoxRWLogin.Text);
                            if (arrLogin != null)
                            {
                                eerror = _reader.Login(arrLogin);
                                if (eerror != EERROR.ER_OK)
                                    MessageBox.Show(string.Format("An error happens by logging to the tag [{0}]", eerror));
                            }
                        }
                        if (eerror == EERROR.ER_OK)
                        {
                            // read data information according to offset and bytes fields
                            byte[] arrData = null;
                            eerror = _reader.ReadData(Convert.ToUInt32(strOffset), Convert.ToUInt32(strNrBytes),
                                                      ref arrData);
                            if (eerror == EERROR.ER_OK)
                            {
                                _textBoxRWData.Text = HexToString(arrData).ToUpper();
                            }
                            else
                                MessageBox.Show(string.Format("An error happens by reading the tag data [{0}]", eerror));
                        }
                    }

                }
                else MessageBox.Show("Error getting card id");
            }
            else MessageBox.Show("Offset & Nr bytes have to be filled");

            _buttonRWRead.Enabled = true;
        }

        private void _buttonRWWrite_Click(object sender, EventArgs e)
        {
            try
            {
                _buttonRWWrite.Enabled = false;

                // check validatidy of parameters
                string strOffset = _textBoxRWOffset.Text;
                UInt32 uintOffset = Convert.ToUInt32(strOffset);
                string strNrBytes = _textBoxRWNrBytes.Text;
                UInt32 uintNrBytes = Convert.ToUInt32(strNrBytes);
                byte[] dataToWrite = HexStringToByteArray(_textBoxRWData.Text);

                if(_textBoxRWData.Text.Length != (uintNrBytes * 4 * 2)) throw new Exception("Invalid data length. Nb of characters should be Nb Bytes * 4 (bytes per block) * 2 (for hexadecimal fornat)");

                if ((strOffset.Length > 0) && (strNrBytes.Length > 0))
                {
                    byte[] arrCardID = null;
                    EERROR eerror;

                    // get tag ID - 's' command
                    if (_reader.GetCardID(ref arrCardID) == EERROR.ER_OK)
                    {
                        // there is no tag in the field
                        if (arrCardID.Length == 1 && (char)arrCardID[0x00] == 'N')
                        {
                            throw new Exception("No tag in the field");
                        }
                        // perform login if the textbox is filled with the password
                        if (_textBoxRWLogin.Text.Length > 0)
                        {
                            byte[] arrLogin = HexStringToByteArray(_textBoxRWLogin.Text);
                            if (arrLogin != null)
                            {
                                eerror = _reader.Login(arrLogin);
                                if (eerror != EERROR.ER_OK)
                                    throw new Exception(string.Format("An error happens by logging to the tag [{0}]", eerror));
                            }
                        }

                        // write data information according to offset, bytes and RWdata fields
                        eerror = _reader.WriteData(uintOffset, uintNrBytes, dataToWrite);
                        if (eerror == EERROR.ER_OK)
                        {
                            _textBoxRWData.Text = "Write success";
                        }
                        else throw new Exception(string.Format("An error happens by writing the tag data [{0}]", eerror));
                    }
                    else throw new Exception("Error getting card id");

                }
                else throw new Exception("Offset & Nr bytes have to be filled");
            }
            catch (Exception exception)
            {
                MessageBox.Show(exception.Message);
            }
            finally
            {
                _buttonRWWrite.Enabled = true;
            }
        }

        private void _textBoxRWOffset_KeyPress(object sender, KeyPressEventArgs e)
        {
            e.Handled = !char.IsDigit(e.KeyChar) && !char.IsControl(e.KeyChar);
        }

        private void _textBoxRWNrBytes_KeyPress(object sender, KeyPressEventArgs e)
        {
            e.Handled = !char.IsDigit(e.KeyChar) && !char.IsControl(e.KeyChar);
        }

        private void _textBoxRWLogin_KeyPress(object sender, KeyPressEventArgs e)
        {
            if (!(Regex.IsMatch(e.KeyChar.ToString(), "^[0-9a-fA-F]+$")))
                e.Handled = true;
        }

        private void _buttonSend_Click(object sender, EventArgs e)
        {
            _buttonSend.Enabled = false;
            string strCommand = _textBoxCommand.Text;
            string strData = _textBoxData.Text;
            if ((strCommand.Length > 0))
            {
                byte[] arrCmdIn = StringToByteArray(strCommand);
                byte[] arrDataIn = null;
                
                // if data available
                if (strData.Length > 0) arrDataIn = StringToByteArray(strData);

                Stopwatch stopwatch = new Stopwatch();
                stopwatch.Start();
                // send the protocol command to the reader and get the answer with 10 retry = 1 sec time out for read
                byte[] arrDataOut = null;
                EERROR eerror = _reader.Protocol(arrCmdIn, arrDataIn, ref arrDataOut, 10);
                stopwatch.Stop();

                // manage error
                if (eerror != EERROR.ER_OK)
                {
                    _textBoxInfo.ForeColor = Color.Red;
                    _textBoxInfo.Text = "Error executing the command: " + eerror.ToString();
                }
                else // show information 
                {
                    _textBoxInfo.ForeColor = Color.Black;
                    _textBoxInfo.Text = ">> " + strCommand + strData + Environment.NewLine;
                    //_textBoxInfo.Text += "<< [hex]" + HexToString(arrDataOut) + Environment.NewLine;
                    _textBoxInfo.Text += "<< " + Encoding.UTF8.GetString(arrDataOut, 0, arrDataOut.Length) + Environment.NewLine;
                    _textBoxInfo.Text += string.Format(" In {0} ms" , stopwatch.ElapsedMilliseconds) + Environment.NewLine;
                }

                // as multiple and wrong commands can be generated on Protocol panel, we clear the buffer after the first answer
                byte[] arrData = null;
                do
                {
                    eerror = _reader.ReadContinuesModeData(ref arrData);
                } while (arrData != null && eerror == EERROR.ER_OK);
            }
            else
            {
                _textBoxInfo.ForeColor = Color.Orange;
                _textBoxInfo.Text = "Command has to be filled";
            }
            _buttonSend.Enabled = true;
        }

    }
}